-------------------------------------------------------------------------------
--                                                                           --
--  FILE          :  univ_sim.vhd                                            --
--  Related FILEs :  seuilleur.vhd   : composant  tester                    --
--                   bloc_entree.txt : fichier de donnees d'entree           --
--                   bloc_sortie.txt : fichier de donnees de sortie          --
--                                                                           --
--  Author(s)     :  E. BOUTILLON                                            --
--                                                                           --
--  Organization  :  IUP LORIENT, LICENCE GEII                               --
--  Project       :  Projet VHDL                                             --
--                                                                           --
--  Creation Date :  25.01.2001                                              --
--                                                                           --
--  Simulator     :  Model Sim                                               --
--                                                                           --
--  Description   :  Environnement de simulation du projet VHDL              --
-------------------------------------------------------------------------------
--                                History                                    --
-------------------------------------------------------------------------------
-- Version 1 : 25/01/2001 Creation                                           --
-------------------------------------------------------------------------------
LIBRARY std;
USE std.textio.all; 
LIBRARY IEEE;
USE IEEE.std_logic_1164.all;

-------------------------------------------------------------------------------
-- declaration de l'entite vide
-------------------------------------------------------------------------------
ENTITY univ_sim IS

END univ_sim;   


-------------------------------------------------------------------------------
-- Architecture de simulation du composant seuillage
-------------------------------------------------------------------------------
ARCHITECTURE sim OF univ_sim IS

-------------------------------------------------------------------------------
-- Declaration du composant seuillage
-------------------------------------------------------------------------------
COMPONENT seuilleur 
	PORT (
	rst       : IN  bit;                      -- signal reset actif a 0
        clk       : IN  bit;                      -- signal d'horloge   
        data_in   : INOUT  std_logic_vector(7 DOWNTO 0);  -- donnees entrees
        data_in_v : IN  bit;                      -- entree valide  
        start     : IN  bit;                      -- debut des calculs
        data_out_r: IN bit;                       -- requete lecture donnee
        end_c     : OUT bit                       -- fin des calculs
        );
END COMPONENT;


-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- Declaration de fonctions de conversions
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- FUNCTION int2vec : Conversion entier en vecteur 8 bits
-------------------------------------------------------------------------------
FUNCTION int2vec(a : NATURAL) RETURN std_logic_vector IS

VARIABLE r : std_logic_vector( 7 DOWNTO 0);
VARIABLE step, aa  : natural := 128;

BEGIN
	aa := a;
	r := "00000000";
	ASSERT (a < 256) REPORT "valeur lue dans fichier non valide" SEVERITY FAILURE;
	
	FOR I in 7 DOWNTO 0 LOOP
		IF (aa > (step - 1)) THEN 
			aa := aa - step; 
			r(I) := '1'; 
		END IF;
		step := step/2;
	END LOOP;
	RETURN (r);
END int2vec;

-------------------------------------------------------------------------------
-- FUNCTION int2vec : Conversion vecteur 8 bits en entier
-------------------------------------------------------------------------------
FUNCTION vec2int(r : std_logic_vector(7 DOWNTO 0)) RETURN NATURAL IS

VARIABLE a : NATURAL := 0;
VARIABLE step  : NATURAL := 1;

BEGIN
	
	calcul : FOR I in 0 TO 7 LOOP
		IF (r(I) = '1') THEN 
			a := a + step; 
		ELSIF (r(I) /= '0') THEN
			a := 999;
			EXIT calcul;
		END IF;		
		step := step*2;
	END LOOP;
	RETURN (a);
END vec2int;

-------------------------------------------------------------------------------
-- Declaration des signaux
-------------------------------------------------------------------------------
SIGNAL rst       : bit;                      -- signal reset actif a 0
SIGNAL clk       : bit;                      -- signal d'horloge   
SIGNAL data_in   : std_logic_vector(7 DOWNTO 0);   -- donnees entrees
SIGNAL data_in_v : bit;                      -- entree valide  
SIGNAL start     : bit;                      -- debut des calculs
SIGNAL data_out_r: bit;                       -- requete lecture donnee
SIGNAL end_c     : bit;                       -- fin des calculs
SIGNAL cela_repart : bit;                    -- pour dialogue entre ecriture/lecture


------------------------------------------------------------------------------
-- AU BOULOT
------------------------------------------------------------------------------
BEGIN

   ---------------------------------------------------------------
   -- Process d'horloge
   ---------------------------------------------------------------
   horloge :    PROCESS
   BEGIN
      clk <='1';
      WAIT FOR 10 ns;
      clk <= '0';
      WAIT FOR 10 ns;
   END PROCESS horloge;

   ---------------------------------------------------------------
   -- Process gestion de la lecture de trois blocs dans le fichier  
   ---------------------------------------------------------------
   lecture : PROCESS

   FILE fich         : TEXT IS IN "bloc_entree.txt";
   VARIABLE ligne    : LINE;
   VARIABLE m        : natural;   
 
   BEGIN
	-----------------------------
	-- Initialisation des signaux
        data_in_v  <= '0';
	start      <= '0';
	data_in    <= (OTHERS => 'Z');

	-------------------------------
	-- On attend avant de commencer
	rst <='0';
	WAIT FOR 47 ns;
 	rst <= '1'; 
	WAIT FOR 100 ns;
	WAIT UNTIL clk'EVENT and clk='1';                 
      
	--------------------------------------------
	-- Premier test : lecture simple des donnes
	--------------------------------------------
        Premier_test_L : FOR k IN 1 TO 256 LOOP
       		------------------------
        	-- Lecture ligne
        	READLINE(fich, ligne);

            	-----------------------
            	-- lecture de l'elements de la ligne
            	-- affichage + signaux controles 
            	READ(ligne,m);
            	data_in   <= INT2VEC(m);               
	    	data_in_v <= '1';            
	    	WAIT UNTIL clk'event and clk='1';
	END LOOP premier_test_L;
	data_in_v <= '0';
	data_in <= "ZZZZZZZZ";
        --------------------
        -- l'ensemble du bloc est rentr, 
	-- on attend avant mettre start a 1
	WAIT UNTIL clk'event and clk='1';
	WAIT UNTIL clk'event and clk='1';
	start <= '1';
	WAIT UNTIL clk'event and clk='1';
	start <= '0';

	-- On attend le passage  1 de cela_retard pour continuer le calcul
	WAIT UNTIL cela_repart = '1';

	---------------------------------------------------
	-- Deuxime test : lecture non continue des donnes
	---------------------------------------------------
        Deuxieme_test_L : FOR k IN 1 TO 256 LOOP
     		------------------------
       		-- Lecture ligne
             	READLINE(fich, ligne);

            	-----------------------
            	-- lecture de l'elements de la ligne
            	-- affichage + signaux controles 
            	READ(ligne,m);
            	data_in   <= INT2VEC(m);               
	    	data_in_v <= '1';            
	    	WAIT UNTIL clk'event and clk='1';
	    
	    	-----------------------
	    	-- remise a zero des signaux
	
            	data_in   <= (OTHERS => 'Z');               
	    	data_in_v <= '0';            

		------------------------
		-- Quelques cycles pour ne pas etre regulier
		IF (k MOD 4) = 0 THEN
			WAIT UNTIL clk'event and clk='1';
		END IF;
		IF (k MOD 7) = 0 THEN
			WAIT UNTIL clk'event and clk='1';
		END IF;

        END LOOP Deuxieme_test_L;
	WAIT UNTIL clk'event and clk='1';
	start <= '1';
	WAIT UNTIL clk'event and clk='1';
	start <= '0';
	-- On attend le passage  1 de cela_retart pour continuer le calcul
	WAIT UNTIL cela_repart = '1';

 	----------------------------------------------------------------
	-- Troisieme test : verification des modes pathologique :
	--          passage de reset  0 aprs 40 ecritures
	--          reprise normale de l'ecriture pendant 40 cycles puis 
	--          passage du bit start  1 => Doit generer un warning.
	----------------------------------------------------------------
        Troisieme_test_reset : FOR k IN 1 TO 40 LOOP
       		------------------------
              	-- Lecture ligne
             	READLINE(fich, ligne);

            	-----------------------
            	-- lecture de l'elements de la ligne
            	-- affichage + signaux controles 
            	READ(ligne,m);
            	data_in   <= INT2VEC(m);               
	    	data_in_v <= '1';            
	    	WAIT UNTIL clk'event and clk='1';
	    	data_in_v <= '0';            

        END LOOP Troisieme_test_reset;
	data_in   <= ("ZZZZZZZZ");
	WAIT FOR 2 ns;
	rst <= '0';
	WAIT FOR 100 ns;
	rst <= '1';
        Troisieme_test_start : FOR k IN 1 TO 40 LOOP
       		------------------------
              	-- Lecture ligne
             	READLINE(fich, ligne);

            	-----------------------
            	-- lecture de l'elements de la ligne
            	-- affichage + signaux controles 
            	READ(ligne,m);
            	data_in   <= INT2VEC(m);               
	    	data_in_v <= '1';            
	    	WAIT UNTIL clk'event and clk='1';
        END LOOP Troisieme_test_start;
	data_in   <= ("ZZZZZZZZ");
	WAIT UNTIL clk'event and clk='1';
	start <= '1';
	WAIT UNTIL clk'event and clk='1';
	start <= '0';
	WAIT UNTIL cela_repart = '1';
	WAIT UNTIL clk'event and clk='1';
	start <= '1';
	WAIT UNTIL clk'event and clk='1';
	start <= '0';
	WAIT UNTIL clk'event and clk='1';
	WAIT UNTIL clk'event and clk='1';
	ASSERT FALSE REPORT "Simulation termine normalement";
	WAIT; -- Il n'y a plus rien a tester
  END PROCESS Lecture;

    
   ---------------------------------------------------------------
   -- Process gestion de l'ecriture d'un bloc 
   ---------------------------------------------------------------
   Ecriture : PROCESS

   FILE fich         : TEXT IS OUT "bloc_sortie.txt";
   VARIABLE ligne    : LINE;
   VARIABLE m        : integer;   
 
   BEGIN
	-----------------------------
	-- Initialisation des signaux
      	cela_repart <= '0';
	data_out_r <= '0';
	----------------------------
	-- On attend un nouveau bloc a ecrire
	WAIT UNTIL end_c = '1';
	
	WAIT UNTIL clk'EVENT and clk='1';                 
      
	----------------------------------------
	-- Premiere boucle d'ecriture du fichier
	-----------------------------------------
 	Premier_test_E : FOR k IN 1 TO 256 LOOP
		data_out_r <= '1';
        	WAIT UNTIL clk'EVENT and clk='1';                 				
                m := VEC2INT(data_in);
		data_out_r <= '0';
	        WRITE(ligne, m);
	        WRITELINE(fich,ligne);
        END LOOP Premier_test_E;
		
	------------------------
	-- La lecture est termine.
	-- On positione ca_repart a 1 pour activer le bloc lecture	
	WAIT UNTIL clk'event and clk='1';
	cela_repart <= '1';
	WAIT UNTIL clk'event and clk='1';
	cela_repart <= '0';


	----------------------------------------
	-- Deuxieme boucle d'ecriture du fichier
	-- On attend tout d'abord que le 2ieme bloc
	-- soit traite.
	-----------------------------------------
	WAIT UNTIL end_c = '1';

 	Deuxieme_test_E : FOR k IN 1 TO 256 LOOP
		data_out_r <= '1';
        	WAIT UNTIL clk'EVENT and clk='1';                 				
                m := VEC2INT(data_in);
		data_out_r <= '0';
	        WRITE(ligne, m);
	        WRITELINE(fich,ligne);
		------------------------
		-- Quelques cycles pour ne pas etre regulier
		IF (k MOD 6) = 0 THEN
			WAIT UNTIL clk'event and clk='1';
		END IF;
		IF (k MOD 5) = 0 THEN
			WAIT UNTIL clk'event and clk='1';
		END IF;
        END LOOP Deuxieme_test_E;
		
	------------------------
	-- La lecture est termine.
	-- On positione ca_repart a 1 pour activer le bloc lecture	
	WAIT UNTIL clk'event and clk='1';
	cela_repart <= '1';
	WAIT UNTIL clk'event and clk='1';
	cela_repart <= '0';

	----------------------------------------
	-- Troisieme boucle d'ecriture du fichier
	-- seul 20 donnes sont lues avant de faire
	-- passer le signal start  1.
	----------------------------------------
	-- On attend tout d'abord que le 2ieme bloc
	-- soit traite.
	-----------------------------------------
	WAIT UNTIL end_c = '1';

 	Troisieme_test_E : FOR k IN 1 TO 20 LOOP
		data_out_r <= '1';
        	WAIT UNTIL clk'EVENT and clk='1';                 				
                m := VEC2INT(data_in);
		data_out_r <= '0';
	        WRITE(ligne, m);
	        WRITELINE(fich,ligne);
		------------------------
		-- Quelques cycles pour ne pas etre regulier
		IF (k MOD 6) = 0 THEN
			WAIT UNTIL clk'event and clk='1';
		END IF;
		IF (k MOD 5) = 0 THEN
			WAIT UNTIL clk'event and clk='1';
		END IF;
        END LOOP Troisieme_test_E;
		
	------------------------
	-- La lecture est termine.
	-- On positione ca_repart a 1 pour activer le bloc lecture	
	WAIT UNTIL clk'event and clk='1';
	cela_repart <= '1';
	WAIT UNTIL clk'event and clk='1';
	cela_repart <= '0';
	WAIT;      -- On tue le process
    
END PROCESS Ecriture;

   ---------------------------------------------------------------
   -- Instanciation du composant seuillage
   ---------------------------------------------------------------
  partie_operative : seuilleur PORT MAP (
	rst       => rst,
        clk       => clk,
        data_in   => data_in,
        data_in_v => data_in_v,
        start     => start,
        data_out_r=> data_out_r,
        end_c     => end_c
	);

END sim;










